import os
import sys
import csv
import requests
import osmium

# --- 1. KONFIGURĀCIJA ---

# Lejupielādes saite un faila nosaukums
URL = 'https://download.geofabrik.de/europe/latvia-latest.osm.pbf'
PBF_FILENAME = 'latvia-latest.osm.pbf'

# Kategoriju definīcijas: CSV fails -> {taga atslēga -> {taga vērtību kopa}}
# Šādi tiek definēts, kurus objektus likt kurā CSV failā.
CATEGORIES = {
    'bar.csv': {
        'amenity': {'bar', 'pub', 'nightclub'}
    },
    'cofe.csv': {
        'amenity': {'cafe'}
    },
    'food.csv': {
        'amenity': {'restaurant', 'fast_food', 'food_court'}
    },
    'frizieri.csv': {
        'shop': {'hairdresser'}
    }
}


# --- 2. PALĪGFUNKCIJAS ---

def download_file(url, filename):
    """Lejupielādē failu no URL, rādot progresu."""
    print(f"Sāk lejupielādēt failu: {filename}")
    try:
        with requests.get(url, stream=True, timeout=30) as r:
            r.raise_for_status()
            total_size = int(r.headers.get('content-length', 0))
            block_size = 8192
            downloaded = 0
            with open(filename, 'wb') as f:
                for chunk in r.iter_content(chunk_size=block_size):
                    f.write(chunk)
                    downloaded += len(chunk)
                    done = int(50 * downloaded / total_size) if total_size > 0 else 0
                    # Rādām progresu megabaitos
                    progress_mb = downloaded / (1024*1024)
                    total_mb = total_size / (1024*1024)
                    sys.stdout.write(f"\r[{'=' * done}{' ' * (50-done)}] {progress_mb:.2f} MB / {total_mb:.2f} MB")
                    sys.stdout.flush()
        print("\nLejupielāde pabeigta.")
        return True
    except requests.exceptions.RequestException as e:
        print(f"\nKĻŪDA: Neizdevās lejupielādēt failu. {e}")
        return False

def write_csv_files(results_data):
    """Ieraksta sagrupētos datus attiecīgajos CSV failos."""
    print("-" * 30)
    print("Sāk ģenerēt CSV failus...")
    total_written_rows = 0

    for filename, rows in results_data.items():
        if not rows:
            print(f"  -> Fails '{filename}' netiek veidots, jo nav atrasts neviens derīgs ieraksts.")
            continue

        written_rows = 0
        print(f"  -> Raksta failu '{filename}' ({len(rows)} ieraksti)...")
        try:
            with open(filename, 'w', encoding='utf-8', newline='') as outfile:
                writer = csv.writer(outfile)
                writer.writerow(['point', 'name'])
                for row_data in rows:
                    point_str = f"POINT({row_data['lon']} {row_data['lat']})"
                    writer.writerow([point_str, row_data['name']])
                    written_rows += 1
            total_written_rows += written_rows
            print(f"     '{filename}' veiksmīgi izveidots ar {written_rows} rindām.")
        except IOError as e:
            print(f"KĻŪDA, rakstot failu '{filename}': {e}")
    
    print("-" * 30)
    print(f"Darbs pabeigts! Kopā visos CSV failos ierakstītas {total_written_rows} rindas.")


# --- 3. PYOSMIUM APSTRĀDĀTĀJS ---

class PlaceHandler(osmium.SimpleHandler):
    def __init__(self):
        super(PlaceHandler, self).__init__()
        # Datu struktūra rezultātu glabāšanai, sadalīta pa CSV failiem
        self.results = {filename: [] for filename in CATEGORIES}

    def node(self, n):
        """Šī funkcija tiek izsaukta katram mezglam (node) PBF failā."""
        
        # Apstrādājam tikai objektus, kuriem ir nosaukums
        if 'name' in n.tags:
            # Pārbaudām, vai objekts atbilst kādai no mūsu kategorijām
            for filename, rules in CATEGORIES.items():
                for tag_key, tag_values in rules.items():
                    if tag_key in n.tags and n.tags[tag_key] in tag_values:
                        # Ja atrasta atbilstība, saglabājam datus
                        self.results[filename].append({
                            'name': n.tags['name'],
                            'lon': n.location.lon,
                            'lat': n.location.lat
                        })
                        # Pārtraucam ciklu, lai viens objekts netiktu pievienots vairākām kategorijām
                        return


# --- 4. GALVENĀ PROGRAMMAS PLŪSMA ---

if __name__ == "__main__":
    # 1. solis: Lejupielādējam PBF failu
    if not os.path.exists(PBF_FILENAME):
        if not download_file(URL, PBF_FILENAME):
            sys.exit(1) # Beidzam darbu, ja lejupielāde neizdevās
    else:
        print(f"Fails '{PBF_FILENAME}' jau pastāv, lejupielāde tiek izlaista.")

    # 2. solis: Apstrādājam PBF failu ar Osmium
    print("-" * 30)
    print(f"Sāk apstrādāt failu '{PBF_FILENAME}' ar pyosmium...")
    
    handler = PlaceHandler()
    try:
        # Palaižam apstrādi, norādot, ka nepieciešamas koordinātas (locations=True)
        handler.apply_file(PBF_FILENAME, locations=True)
        print(f"Apstrāde pabeigta.")
    except Exception as e:
        print(f"KĻŪDA, apstrādājot PBF failu: {e}")
        sys.exit(1)

    # 3. solis: Ierakstām rezultātus CSV failos
    write_csv_files(handler.results)

    # 4. solis: Tīrīšana - izdzēšam lielo lejupielādēto failu
    try:
        print(f"Tīra pagaidu failu '{PBF_FILENAME}'...")
        os.remove(PBF_FILENAME)
        print("Tīrīšana pabeigta.")
    except OSError as e:
        print(f"Kļūda, dzēšot failu {PBF_FILENAME}: {e}")